#ifndef RAYTRACING_GLSL
#define RAYTRACING_GLSL

#include <primitives.glsl>

#define MAXDISTANCE 180.
#define EPSILON 0.00001
#define M_PI 3.1415926535897932384626433832795
#define MORPH_VM_MEMORY 20

struct Ray {
    vec3 origin;
    vec3 direction;
};

vec4 sdfScene(vec3 point);

///////////
// Transformations
mat4 translate4d(vec3 t) {
    return mat4(1.0, 0.0, 0.0, 0.0,
                0.0, 1.0, 0.0, 0.0,
                0.0, 0.0, 1.0, 0.0,
                t.x, t.y, t.z, 1.0 );
}

mat4 rotate4d(in vec3 a, const in float r) {
    a = normalize(a);
    float s = sin(r);
    float c = cos(r);
    float oc = 1.0 - c;
    vec4 col1 = vec4(oc * a.x * a.x + c, oc * a.x * a.y + a.z * s, oc * a.z * a.x - a.y * s, 0.0);
    vec4 col2 = vec4(oc * a.x * a.y - a.z * s, oc * a.y * a.y + c, oc * a.y * a.z + a.x * s, 0.0);
    vec4 col3 = vec4(oc * a.z * a.x + a.y * s, oc * a.y * a.z - a.x * s, oc * a.z * a.z + c, 0.0);
    vec4 col4 = vec4(0.0, 0.0, 0.0, 1.0);
    return mat4(col1, col2, col3, col4);
}

/////////////
// SDF Operations
float steps(in float point, float stepSize) {
    return round(point / stepSize) * stepSize;
}

float smoothMin(float d1, float d2, float k) {
    float h = max(k - abs(d1 - d2), 0.0) / k;
    return min(d1, d2) - h * h * h * k * (1.0 / 6.0);
}

vec4 opU(vec4 d1, vec4 d2) {
    return d1.x < d2.x ? d1 : d2;
}

///////////
// Random
vec4 random;

void shuffle() {
    random = fract(1e4 * sin(random) + random.wxyz + 0.5);
}

void initRandom(vec2 normalizedPixel, uint frame) {
    random = vec4(normalizedPixel, frame, 1);

    for(int i = 0; i < 16; i++)
        shuffle();
}

vec3 cosineWeightedRandomHemisphereDirection(const vec3 n) {
	vec3  uu = normalize(cross(n, vec3(0.0, 1.0, 1.0)));
	vec3  vv = cross(uu, n);
	float ra = sqrt(random.y);
    float rb = 6.2831 * random.x;
	float rx = ra * cos(rb); 
	float ry = ra * sin(rb);
	float rz = sqrt( 1.0 - random.y);
	vec3  rr = vec3( rx * uu + ry * vv + rz * n );

    return normalize(rr);
}

///////////
// Marching
#define MAX_STEPS 100

vec3 computeNormal(vec3 point) {
    vec3 epsilon = vec3(EPSILON, 0.0, 0.0);

    float dx = sdfScene(point + epsilon.xyy).x - sdfScene(point - epsilon.xyy).x;
    float dy = sdfScene(point + epsilon.yxy).x - sdfScene(point - epsilon.yxy).x;
    float dz = sdfScene(point + epsilon.yyx).x - sdfScene(point - epsilon.yyx).x;

    return normalize(vec3(dx, dy, dz));
}

vec4 march(Ray ray) {
    vec4 result = vec4(0.0);
    int i;

    for(i = 0; i < MAX_STEPS; i++) {
    	vec3 pivot = ray.origin + ray.direction * result.x; 
        vec4 sdf = sdfScene(pivot);
        result.x += sdf.x;
        result.yzw = sdf.yzw;

        if (result.x > MAXDISTANCE) {
            //TODO: this is a hack
            result.yzw = vec3(0.0);
            break;
        }

        if (result.x < EPSILON)
            break;
    }

    return result;
}

#endif
